const express = require("express");
const cors = require("cors");
const morgan = require("morgan");
const http = require("http");
const path = require("path");
const fileUpload = require("express-fileupload");
const helmet = require("helmet");
const moment = require("moment-timezone");
const colors = require("colors");

const { PORT } = require("./config") || 5000;
const api = require("./routes/router.js");
const { authMiddleware } = require("./middleware/authMiddleware.js");
const { errorHandler } = require("./helper/logger.helper.js");

// const appRouter = require("./Routes")

const app = express();

// List of allowed frontend origins
const allowedOrigins = [
  "http://localhost:5173",
  "http://localhost:5174",
  "https://opex3-shanfoods.oee-tracker.com",
  "https://leds-shanfoods.oee-tracker.com",
];

// CORS config with multiple origins and exposed header
app.use(
  cors({
    origin: function (origin, callback) {
      // Allow non-browser tools (like Postman) where origin is undefined
      if (!origin) return callback(null, true);
      if (allowedOrigins.includes(origin)) {
        return callback(null, true);
      }
      return callback(new Error("Not allowed by CORS"));
    },
    credentials: true,
    exposedHeaders: ["new-token"], // <--- THIS is key for your interceptor
  })
);

app.use(morgan("dev"));
app.use(express.urlencoded({ extended: false }));
app.use(express.json({ extended: false }));
app.use(express.json());
app.use(express.static("public"));
app.use(fileUpload({ safeFileNames: true, preserveExtension: true }));

//===> Helmet for security
app.use(helmet({
  contentSecurityPolicy: {
    useDefaults: true,
    directives: {
      "img-src": ["'self'", "data:", "blob:", "http://localhost:5001"],
    },
  },
  crossOriginResourcePolicy: { policy: "cross-origin" },
}));

// Apply the rate limiting middleware to all requests.
// app.use(limiter);
app.use("/uploads", express.static("uploads"));
app.use("/uploads/images", express.static("uploads/images"));


app.use("/api", authMiddleware, api);

app.use(errorHandler)

app.use(
  "/",
  express.static(path.resolve(path.join(__dirname, "./client/dist")))
);

app.use(
  "*",
  express.static(path.resolve(path.join(__dirname, "./client/dist")))
);

const server = http.createServer(app);
// const PORT = process.env.PORT || 5000;
server.listen(PORT, () => {
  console.log(`Server is running on: http://localhost:${PORT}`.bgCyan.green.bold);
});
