const { sendMail } = require("../services/MAIL.js");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const {
  sendErrorResponse,
  sendResponse,
  generatePasswordResetToken,
} = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { JWT_SECRET } = require("../config/index.js");
const { performQuery } = require("../utils/db.js");
const moment = require("moment-timezone");
const { generateOTP } = require("../services/OTP.js");
const { getSystemTime } = require("../functions/getTimezone.js");

// API to handle user sign-up request
module.exports.signUp = async (req, res) => {
  try {
    // Extracting name, email, and password from request body
    const { name, email, password } = req.body;

    // Validate the name, email, and password
    if (!name || !email || !password) {
      const error = new Error("Name, Email and password are required");
      return sendErrorResponse(
        res,
        error,
        "Name, Email and password are required"
      );
    }

    // Check if the user already exists
    const selectUserQuery = `
      SELECT * 
      FROM ${tables.users} 
      WHERE email = ? AND is_deleted = 0`;
    const selectParams = [email];

    // Perform the database query
    const existingUser = await performQuery(selectUserQuery, selectParams);
    if (existingUser?.length) {
      return sendErrorResponse(
        res,
        "User already exists",
        "User already exists",
        409
      );
    }

    // Hash the password
    const hashedPassword = await bcrypt.hash(password, 10);

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert the new user into the database
    const InsertUser = await performQuery(`INSERT INTO ${tables.users} SET ?`, {
      name: name,
      email: email,
      password: hashedPassword,
      is_new: 0,
      created_at: currentTime,
      updated_at: currentTime,
    });

    // Send a response indicating successful sign-up
    return sendResponse(
      res,
      { userId: InsertUser.insertId },
      "Sign Up successful",
      201
    );
  } catch (error) {
    return sendErrorResponse(res, error, "Error while Sign Up");
  }
};

// API to handle user login request
module.exports.login = async (req, res) => {
  try {
    // Extracting email and password from request body
    const { email, password } = req.body;
    // Validate the email and password
    if (!email || !password) {
      const error = new Error("Email and password are required");
      return sendErrorResponse(res, error, "Email and password are required");
    }

    // Find the user by email address
    const selectUserQuery = `
      SELECT * 
      FROM ${tables.users} 
      WHERE email = ? AND is_deleted = 0`;
    const selectParams = [email];

    // Perform the database query
    const user = await performQuery(selectUserQuery, selectParams);
    const userRecord = user?.[0];

    // If the user does not exist, send an error response
    if (!user?.length) {
      return sendErrorResponse(res, {}, "Invalid email or password", 402);
    }
    // If the user does not exist, send an error response
    if (!userRecord?.is_active) {
      return sendErrorResponse(res, {}, "User Not Active", 402);
    }

    // Compare the password from the request with the user's password
    const isPasswordValid = await bcrypt.compare(
      password,
      userRecord?.password
    );

    // If the password is invalid, send an error response
    if (!isPasswordValid) {
      return sendErrorResponse(res, {}, "Invalid email or password", 403);
    }

    // If the user is_new is 1 then send an error response
    if (userRecord?.is_new) {
      const errorMeassage = "Please change your password to continue";
      return sendErrorResponse(res, errorMeassage, errorMeassage, 403);
    }

    // Generate a JWT token for the user
    const payload = {
      id: userRecord?.id,
      name: userRecord?.name,
      role: userRecord?.role,
      profileUrl: userRecord?.profileUrl,
      is_admin: userRecord?.is_admin,
      iat: Math.floor(Date.now() / 1000),
    };
    const token = await jwt.sign(payload, JWT_SECRET, { expiresIn: "6h" });

    // Send the token back to the user
    return sendResponse(
      res,
      { token, user: { id: userRecord?.id, name: userRecord?.name } },
      "Login successful",
      200
    );
  } catch (error) {
    return sendErrorResponse(res, error, "Error while login");
  }
};

// API to handle set password request (When New User Login First Time user has to set password)
module.exports.setPassword = async (req, res) => {
  try {
    // get Email and New Password from request body
    const { email, newPassword } = req.body;

    // Validate the email and new password
    if (!email || !newPassword) {
      const errorMeassage = "Email and New Password are required";
      return sendErrorResponse(res, errorMeassage, errorMeassage);
    }

    const selectUserQuery = `SELECT * FROM ${tables.users} WHERE email = ? AND is_deleted = 0`;
    const selectParams = [email];
    const user = await performQuery(selectUserQuery, selectParams);
    const userRecord = user?.[0];

    // If the user does not exist, send an error response
    if (!user) {
      return sendErrorResponse(res, {}, "User does not exist", 404);
    }

    // Hash the new password
    const hashedPassword = await bcrypt.hash(newPassword, 10);

    // Update the user's password and is_new flag in the database
    const updateQuery = `UPDATE ${tables.users} SET password = ?, is_new = 0 WHERE id = ?`;
    const updateParams = [hashedPassword, user[0]?.id];
    await performQuery(updateQuery, updateParams);
    // Generate a JWT token for the user
    
    const payload = {
      id: userRecord?.id,
      name: userRecord?.name,
      role: userRecord?.role,
      profileUrl: userRecord?.profileUrl,
      is_admin: userRecord?.is_admin,
      iat: Math.floor(Date.now() / 1000),
    };
    const token = await jwt.sign(payload, JWT_SECRET, { expiresIn: "6h" });

    // Send a response to the user indicating that the password has been set
    return sendResponse(
      res,
      { token, user: { id: userRecord?.id, name: userRecord?.name } },
      "Password set successfully",
      200
    );
  } catch (error) {
    return sendErrorResponse(res, error, "Error while Set Password");
  }
};

// API to handle change password request
module.exports.changePassword = async (req, res) => {
  try {
    // get User ID, Old Password and New Password from request body
    const { id, oldPassword, newPassword } = req.body;

    // Validate the userId, old password and new password
    if (!id || !oldPassword || !newPassword) {
      const errorMeassage =
        "User ID, Old Password and New Password are required";
      return sendErrorResponse(res, errorMeassage, errorMeassage);
    }

    // Find the user by user ID
    const user = await performQuery(
      `SELECT * FROM ${tables.users} WHERE id = ? AND is_deleted = 0`,
      [id]
    );
    // If the user does not exist, send an error response
    if (!user) {
      return sendErrorResponse(res, {}, "User does not exist", 404);
    }
    const userRecord = user?.[0];

    // Compare the old password from the request with the user's password
    const isOldPasswordValid = await bcrypt.compare(
      oldPassword,
      userRecord?.password
    );

    // If the old password is invalid, send an error response
    if (!isOldPasswordValid) {
      return sendErrorResponse(res, {}, "Invalid old password", 401);
    }

    // Validate the strength of the new password
    const strongPattern =
      /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[!@#$%^&*]).{8,}$/;

    if (!strongPattern.test(newPassword)) {
      const error = new Error("Password is not strong enough");
      return sendErrorResponse(
        res,
        error,
        "Password is not strong enough. It must be at least 8 characters long and include uppercase letters, lowercase letters, numbers, and special characters."
      );
    }

    // Hash the new password
    const hashedPassword = await bcrypt.hash(newPassword, 10);

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update the user's password in the database
    const updateUser = await performQuery(
      `UPDATE ${tables.users} SET ? WHERE id = ?`,
      [
        {
          password: hashedPassword,
          is_new: 0,
          updated_at: currentTime,
          updated_by: req.user?.id,
        },
        id,
      ]
    );

    // Send a response to the user indicating that the password has been changed
    return sendResponse(res, {}, "Password changed successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while Change Password");
  }
};

// API to handle forgot password request
module.exports.forgotPassword = async (req, res) => {
  try {
    // Define the forgot password route
    // Get the email from the request body
    const { email } = req.body;
    if (!email?.length) {
      const error = new Error("Email is required");
      return sendErrorResponse(res, error, "Email is required");
    }

    // Find the user by email address
    const selectUserQuery = `SELECT * FROM ${tables.users} WHERE email = ? AND is_deleted = 0`;
    const selectParams = [email];

    // Perform the database query
    const user = await performQuery(selectUserQuery, selectParams);

    // If the user does not exist, send an error response
    if (!user) {
      return sendErrorResponse(res, {}, "User does not exist", 404);
    }

    // If the user does not exist, send an error response
    if (!user?.[0]?.is_active) {
      return sendErrorResponse(res, {}, "User Not Active", 401);
    }

    // Generate a password OTP
    // const resetToken = await generatePasswordResetToken();
    const resetToken = await generateOTP();

    // Update the user's password OTP in the database
    const updateQuery = `UPDATE ${tables.users} SET password_rt = ?, password_rt_created_at = ? WHERE id = ?`;
    const updateParams = [
      resetToken,
      moment().format("YYYY-MM-DD HH:mm:ss"),
      user[0]?.id,
    ];

    // Perform the database query
    await performQuery(updateQuery, updateParams);

    const emailBody = `
    <!DOCTYPE html>
    <html>
      <head>
         <title>Reset Password</title>
      </head>
      <body style="font-family: Arial, sans-serif; background-color: #f4f4f4; padding: 20px;">
         <div style="max-width: 600px; margin: 0 auto; background-color: #ffffff; padding: 30px; border-radius: 10px;">
            <h1 style="color: #333333;">Hello ${user[0]?.name},</h1>
            <p style="color: #666666;">Your OTP is "<b>${resetToken}</b>".</p>
            <p style="color: #666666;">Please enter this token to Reset your Passowrd.</p>
            <p style="color: #666666;">This token will expire in 30 minutes.</p>
            <p style="color: #666666; font-size: 12px;">If you did not request this token, please ignore this email.</p>
          </div>
      </body>
    </html>
    `;
    // Send the password OTP to the user's email address
    await sendMail(
      email,
      "Reset your password",
      emailBody
      // `Your password OTP is: ${resetToken}`
    );

    // Send a response to the user indicating that the password OTP has been sent
    return sendResponse(
      res,
      {},
      "Password OTP sent to your email address",
      200
    );
  } catch (error) {
    return sendErrorResponse(res, error, "Error while forgot password");
  }
};

// API to handle reset password request
module.exports.resetPassword = async (req, res) => {
  try {
    // Define the reset password route
    // Get the password OTP and new password from the request body
    const { resetToken, newPassword } = req.body;
    if (!resetToken || !newPassword) {
      const error = new Error("NewPassword and ResetToken are required");
      return sendErrorResponse(
        res,
        error,
        "NewPassword and ResetToken are required"
      );
    }

    // Find the user by password OTP
    const selectUserQuery = `SELECT * FROM ${tables.users} WHERE password_rt = ? AND is_deleted = 0`;
    const selectParams = [resetToken];

    // Perform the database query
    const user = await performQuery(selectUserQuery, selectParams);

    // If the user does not exist, send an error response
    if (!user) {
      return sendErrorResponse(res, {}, "Invalid password OTP", 401);
    }

    // Check if the password OTP has expired
    const passwordResetExpirationDate = new Date(
      user[0]?.password_rt_created_at
    );
    passwordResetExpirationDate.setMinutes(
      passwordResetExpirationDate.getMinutes() + 30
    );

    if (passwordResetExpirationDate < new Date()) {
      return sendErrorResponse(
        res,
        {},
        "Token Expired, please try again later",
        401
      );
    }

    // Hash the new password
    const hashedPassword = await bcrypt.hash(newPassword, 10);

    // Update the user's password in the database
    const updateQuery = `UPDATE ${tables.users} SET password_rt = NULL, password_rt_created_at = NULL, password = ? WHERE id = ?`;
    const updateParams = [hashedPassword, user[0]?.id];

    // Perform the database query
    await performQuery(updateQuery, updateParams);

    // Send a response to the user indicating that the password has been reset
    return sendResponse(res, {}, "Password reset successfully", 200);
  } catch (error) {
    return sendErrorResponse(res, error, "Error while Password reset");
  }
};
