const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");

module.exports.getDepartments = async (req, res) => {
  try {
    // Get Filter Parameters from Query
    const { id, department } = req.query;

    // Build the base select query
    let selectDepartmentQuery = `SELECT * FROM ${tables.department} WHERE is_deleted = 0`;
    
    // Apply filters if present
    if (id) {
      selectDepartmentQuery += ` AND id=${id}`;
    }

    // Order by id DESC
    selectDepartmentQuery += ` ORDER BY id DESC`;

    // Perform the database query
    const departmentRecords = await performQuery(selectDepartmentQuery);

    // Send the response
    return sendResponse(res, departmentRecords, "Fetch Department data successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching Department");
  }
};

module.exports.createDepartment = async (req, res) => {
  try {
    const { department } = req.body;

    if (!department) {
      return sendErrorResponse(res, "Department Name is required", "Department Name is required");
    }

    // Check if department already exists
    const departmentData = await performQuery(
      `SELECT * FROM ${tables.department} WHERE department = ? and is_deleted = 0`,
      [department]
    );
    if (departmentData?.length) {
      console.log("Department is already in use");
      return sendErrorResponse(res, "Department is already in use", "Department is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
    
    const result = await performQuery(
      `INSERT INTO ${tables.department} SET ? `,
      {
        department,
        created_at: currentTime,
        created_by: req.user?.id || null,
        updated_at: currentTime,
        updated_by: req.user?.id || null,
      }
    );

    // Send the response
    return sendResponse(res, { department_id: result.insertId }, "Department created successfully", 201);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating Department", 409);
  }
};

module.exports.updateDepartment = async (req, res) => {
  try {
    const { id, department, is_active } = req.body;

    // Validate input
    if (!department || !id) {
      const errorMessage = "Department and id is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // check if department already exists
    const departmentData = await performQuery(
      `SELECT * FROM ${tables.department} WHERE department = ? AND id <> ? and is_deleted = 0`,
      [department, id]
    );
    if (departmentData?.length) {
      return sendErrorResponse(res, "Department is already in use", "Department is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update Department
    const result = await performQuery(`UPDATE ${tables.department} SET ? WHERE id = ?`, [{
      department,
      is_active: is_active !== undefined ? is_active : 1,
      updated_at: currentTime,
      updated_by: req.user?.id || null,
    }, id]);
    
    // Send the response
    return sendResponse(res, result, "Department updated successfully", 201);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while updating Department");
  }
};

module.exports.deleteDepartment = async (req, res) => {
  try {
    const { id } = req.query;

    if (!id) {
      return sendErrorResponse(res, "id is required", "Id is required");
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft Delete Department
    const result = await performQuery(`UPDATE ${tables.department} SET ? WHERE id = ?`, [{
      is_deleted: 1,
      updated_at: currentTime,
      updated_by: req.user?.id || null,
    }, id]);

    // Send the response
    return sendResponse(res, result, "Department deleted successfully", 200);
    
  } catch (error) {
    return sendErrorResponse(res, error, "Error while deleting Department");
  }
};
