const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");


module.exports.createDesignation = async (req, res) => {
  try {
    // Extract data from request body
    const { designation} = req.body;

    // Validate required fields
    if (!designation) {
      const error = new Error("designation is required");
      return sendErrorResponse(res, error, "Designationss is required");
    }

    // Check for existing designation
    const existingDesignation = await performQuery(
      `SELECT * FROM ${tables.designations} WHERE designation = ? AND is_deleted = 0`,
      [designation]
    );
    if (existingDesignation?.length) {
      const errorMessage = "designation is already in use";
      return sendErrorResponse(res, errorMessage, errorMessage, 409);
    }
    
    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert new designation into the database
    const result = await performQuery(
      `INSERT INTO ${tables.designations} SET ? `,
      {
        designation,
        created_by: req.user?.id,
        created_at: currentTime,
        updated_by: req.user?.id,
        updated_at: currentTime,
      }
    );

    // Send the response
    return sendResponse(res, { designation_id: result.insertId }, "designation created successfully", 201);

  } catch (error) {
    console.log("Error while creating designation: ", error);
    return sendErrorResponse(res, error, "Error while creating designation", 409);
  }
};

module.exports.updateDesignation = async (req, res) => {
  try {
    // Extract data from request body
    const { id, designation, is_active } = req.body;

    // Validate required fields
    if (!designation || !id) {
      const error = new Error("designation and id is required");
      return sendErrorResponse(res, error, "designation and id is required");
    }

    // Check for existing designation
    const existingDesignation = await performQuery(
      `SELECT * FROM ${tables.designations} WHERE designation = ? AND id <> ? AND is_deleted = 0`,
      [designation, id]
    );
    if (existingDesignation?.length) {
      const errorMessage = "designation is already in use";
      return sendErrorResponse(res, errorMessage, errorMessage, 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update designation in the database
    const result = await performQuery(
      `UPDATE ${tables.designations} SET ? WHERE id = ?`,
      [
        {
          designation,
          is_active: is_active !== undefined ? is_active : 1,
          updated_by: req.user?.id,
          updated_at: currentTime,
        },
        id,
      ]
    );

    // Send the response
    return sendResponse(res, result, "designation updated successfully", 200);

  } catch (error) {
    console.log("Error while Updating Designation: ", error);
    return sendErrorResponse(res, error, "Error while updating designation");
  }
};

module.exports.getDesignations = async (req, res) => {
  try {
    // Get Filter parameters from query
    const { id } = req.query;

    // Build the query
    let query = `SELECT * FROM ${tables.designations} WHERE is_deleted = 0`;

    // Apply filters
    if (id) {
      query += ` AND id = ${id}`;
    }

    // Order by id DESC
    query += ` ORDER BY id DESC`;

    // Perform the database query
    const designationRecords = await performQuery(query);

    // Send the response
    return sendResponse(res, designationRecords, "Fetch designation data successfully", 200);

  } catch (error) {
    console.log("Error while fetching designation: ", error);
    return sendErrorResponse(res, error, "Error while fetching designation");
  }
};

module.exports.deleteDesignation = async (req, res) => {
  try {
    // Get id from query parameters
    const { id } = req.query;

    // Validate id
    if (!id) {
      const error = new Error("Id is required");
      return sendErrorResponse(res, error, "Id is required");
    }
    
    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft delete the designation
    const result = await performQuery(
      `UPDATE ${tables.designations} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_by: req.user?.id,
          updated_at: currentTime,
        },
        id,
      ]
    );

    // Send the response
    return sendResponse(res, result, "designation deleted successfully", 200);

  } catch (error) {
    console.log("Error while deleting designation: ", error);
    return sendErrorResponse(res, error, "Error while deleting designation");
  }
};
