const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");


module.exports.createEquipment = async (req, res) => {
  try {
    // Extract data from the request body
    const { line_id, equipment, owner_id } = req.body;

    // validation
    if (!line_id || !equipment || !owner_id) {
      const errorMessage = "Line ID, Equipment Name and owner_id are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if equipment already exists
    const equipmentData = await performQuery(
      `SELECT * FROM ${tables.equipment} WHERE equipment = ? and line_id = ? and is_deleted = 0`,
      [equipment, line_id]
    );
    if (equipmentData?.length) {
      return sendErrorResponse(res, "Equipment is already in use", "Equipment is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert new equipment record
    const result = await performQuery(
      `INSERT INTO ${tables.equipment} SET ? `,
      {
        line_id: Number(line_id),
        equipment,
        owner_id: Number(owner_id),
        created_at: currentTime,
        created_by: req.user?.id,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }
    );

    // Send the response
    return sendResponse(res, { equipment_id: result.insertId }, "Equipment created successfully", 201);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating Equipment");
  }
};

module.exports.updateEquipment = async (req, res) => {
  try {
    const { id, line_id, equipment, owner_id, is_active } = req.body;

    if (!line_id || !owner_id || !equipment || !id) {
      const errorMessage = "Id, Line ID, Equipment Name and owner_id are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if equipment already exists
    const equipmentData = await performQuery(
      `SELECT * FROM ${tables.equipment} WHERE equipment = ? AND line_id = ? AND id <> ? and is_deleted = 0`,
      [equipment, line_id, id]
    );
    if (equipmentData?.length) {
      return sendErrorResponse(res, "Equipment is already in use", "Equipment is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update equipment record
    const result = await performQuery(
      `UPDATE ${tables.equipment} SET ? WHERE id = ?`,[{
        line_id: Number(line_id),
        equipment,
        owner_id: Number(owner_id),
        is_active: is_active !== undefined ? is_active : 1,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }, id]
    );

    // Send the response
    return sendResponse(res, result, "Equipment updated successfully", 201);

  } catch (error) {
    console.log("error > ", error);
    return sendErrorResponse(res, error, "Error while Updating Equipment");
  }
};

module.exports.getEquipments = async (req, res) => {
  try {
    // Get Filter Parameters from Query
    const { id, department_id, line_id, owner_id } = req.query;

    // Build the base select query
    let selectEquipmentQuery = `
      SELECT 
        equip.id, equip.equipment, equip.line_id, ln.line, ln.department_id, dept.department, equip.owner_id, usr.name AS owner_name, equip.is_active
      FROM ${tables.equipment} equip
      LEFT JOIN ${tables.line} ln ON equip.line_id = ln.id
      LEFT JOIN ${tables.department} dept ON ln.department_id = dept.id
      LEFT JOIN ${tables.users} usr ON equip.owner_id = usr.id 
      WHERE equip.is_deleted = 0`;

    // Apply filters based on query parameters
    if (id) {
      selectEquipmentQuery += ` AND equip.id=${id}`;
    }
    if (department_id) {
      selectEquipmentQuery += ` AND dept.id=${department_id}`;
    }
    if (line_id) {
      selectEquipmentQuery += ` AND ln.id=${line_id}`;
    }
    if (owner_id) {
      selectEquipmentQuery += ` AND usr.id=${owner_id}`;
    }

    // Order by id DESC
    selectEquipmentQuery += ` ORDER BY equip.id DESC`;

    // Perform the database query
    const equipmentRecords = await performQuery(selectEquipmentQuery);

    // Send the response
    return sendResponse(res, equipmentRecords, "Fetch Equipment data successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching Equipment");
  }
};

module.exports.deleteEquipment = async (req, res) => {
  try {
    const { id } = req.query;

    // Validation
    if (!id) {
      const errorMessage = "Equipment ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft delete the equipment
    const result = await performQuery(
      `UPDATE ${tables.equipment} SET ? WHERE id = ?`,
      [{
        is_deleted: 1,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }, id]
    );

    // Send the response
    return sendResponse(res, result, "Equipment deleted successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while deleting Equipment");
  }
};
