const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");


module.exports.createLine = async (req, res) => {
  try {
    // Extract data from the request body
    const { department_id, line, is_leds } = req.body;

    // Validate required fields
    if (!department_id || !line || is_leds === undefined) {
      const errorMessage = "Department_id, line, and is_leds are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for Duplicate Line in the same Department
    const existingLine = await performQuery(
      `SELECT * FROM ${tables.line} WHERE line = ? AND department_id = ? AND is_deleted = 0`,
      [line, department_id]
    );
    if (existingLine?.length) {
      const errorMessage = "Line is already in use";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
    
    const result = await performQuery(
      `INSERT INTO ${tables.line} SET ?`,
      {
        department_id: department_id,
        line: line,
        is_leds: is_leds,
        created_by: req.user?.id,
        created_at: currentTime,
        updated_by: req.user?.id,
        updated_at: currentTime
      }
    );

    // Send success response
    return sendResponse(res, { line_id: result.insertId }, "Line created successfully", 201);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating Line");
  }
};

module.exports.updateLine = async (req, res) => {
  try {
    // Extract data from the request body
    const { id, department_id, line, is_leds, is_active } = req.body;

    // Validate required fields
    if (!id || !department_id || !line || is_leds === undefined ) {
      const errorMessage = new Error("Department_id, id, line, and is_leds are required");
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for Duplicate Line in the same Department
    const existingLine = await performQuery(
      `SELECT * FROM ${tables.line} WHERE line = ? AND department_id = ? AND id <> ? AND is_deleted = 0`,
      [line, department_id, id]
    );
    if (existingLine?.length) {
      const errorMessage = "Line is already in use";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    const result = await performQuery(
    `UPDATE ${tables.line} SET ? WHERE id = ?`, [{
      department_id: department_id,
      line: line,
      is_leds: is_leds,
      is_active: is_active,
      updated_by: req.user?.id,
      updated_at: currentTime
    }, id]);

    // Send success response
    return sendResponse(res, result, "Line updated successfully", 200);

  } catch (error) {
    console.log("Error While Updating Line: ", error);
    return sendErrorResponse(res, error, "Error while Updating Line");
  }
};

module.exports.getLines = async (req, res) => {
  try {
    // Get Filter Parameters from Query
    const { department_id, id, is_leds } = req.query;

    // Build Query
    let query = `
      SELECT 
        line.id, line.department_id, department.department, line.line, line.is_leds, line.is_active
      FROM ${tables.line}
      LEFT JOIN ${tables.department} ON line.department_id = department.id
      WHERE line.is_deleted = 0`;

    // Apply Filter if Any
    if(id){
      query += ` AND line.id = ${id}`;
    }
    if(department_id){
      query += ` AND line.department_id = ${department_id}`;
    }
    if(is_leds !== undefined){
      query += ` AND line.is_leds = ${is_leds}`;
    }

    // Sort Data in DESC order based on ID
    query += ` ORDER BY line.id DESC`;

    // Execute Query
    const lines = await performQuery(query);

    // Send Success Response
    return sendResponse(res, lines, "Lines fetched successfully", 200);

  } catch (error) {
    console.log("Error While Fetching Lines: ", error);
    return sendErrorResponse(res, error, "Error while fetching Line");
  }
};

module.exports.deleteLine = async (req, res) => {
  try {
    const { id } = req.query;

    if (!id) {
      const errorMessage = new Error("Id is required");
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    const result = await performQuery(
      `UPDATE ${tables.line} SET ? WHERE id = ?`,
      [{
        is_deleted: 1,
        updated_by: req.user?.id,
        updated_at: currentTime
      }, id]
    );

    // Send success response
    return sendResponse(res, result, "Line deleted successfully", 200);

  } catch (error) {
    console.log("Error While Deleting Line: ", error);
    return sendErrorResponse(res, error, "Error while deleting Line");
  }
};
