const tables = require("../utils/tables.js");
const { sendResponse, sendErrorResponse } = require("../utils/index.js");
const { performQuery } = require("../utils/db.js");
// const { transformData } = require("../functions/attributeFeedback.transform.js");
const moment = require("moment-timezone");
const { getSystemTime } = require("../functions/getTimezone.js");

//=====> Create (POST) API for Order ========
module.exports.createOrder = async (req, res) => {
  try {
    //===> Get Input From User
    let {line_id_list} = req.body;
    const {
      line_id,
      order_no,
      order_description,
      expected_qty,
      qty_unit,
      product_id,
      expected_start_date,
    } = req.body;

    if(line_id && !line_id_list){
      line_id_list = [line_id];
    }


    // Validate required fields
    if (
      (!line_id_list || line_id_list?.length == 0) ||
      !order_no ||
      !order_description ||
      !expected_qty ||
      !qty_unit ||
      !product_id
    ) {
      const errorMessage = "All fields are required";
      return sendErrorResponse(res, errorMessage, errorMessage, 400);
    }

    // check for duplicate order_no
    const existingOrder = await performQuery(
      `SELECT * FROM ${tables.orders} WHERE order_no = ? and is_deleted = 0`,
      [order_no]
    );
    if (existingOrder.length > 0) {
      const errorMessage = "Order with this Order No already exists";
      return sendErrorResponse(res, errorMessage, errorMessage, 400);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Create Order Against each line_id
    const order_ids = await Promise.all(
      line_id_list?.map(async (line_id) => {
        const createOrder = await performQuery(
          `INSERT INTO ${tables.orders} SET ?`,
          {
            order_no,
            order_description,
            line_id,
            product_id,
            expected_qty,
            qty_unit,
            expected_start_date: expected_start_date || null,
            created_by: req.user?.id,
            created_at: currentTime,
            updated_by: req.user?.id,
            updated_at: currentTime,
          }
        );

        return createOrder.insertId;
      })
    );
    
    // Send success response
    return sendResponse(res, { order_id: order_ids }, "Orders created successfully", 201);

  } catch (error) {
    console.error("Failed to Create Order", error);
    return sendErrorResponse(res, error, "Failed to Create Order", 500);
  }
};

//=====> Select (GET) API for Order ========
module.exports.getOrders = async (req, res) => {
  try {
    // Get Filter Parameters from Query
    const { id, line_id, status, start_date, end_date, is_leds } = req.query;

    let selectQuery = `
      SELECT 
        o.id, o.order_no, o.order_description, 
        o.line_id, l.line, 
        o.product_id, p.product_code, p.product_name, p.feed_rate, p.speed as product_speed, p.feed_rate, 
        o.status, o.expected_qty, o.qty_unit, o.expected_start_date, 
        od.id AS order_detail_id, od.production, od.shipable_product, od.scrap, 
        od.order_start, od.order_end, od.shift_id, s.shift_name,
        u1.name AS created_by, o.created_at, 
        u2.name AS updated_by, o.updated_at
      FROM 
        ${tables.orders} AS o
      LEFT JOIN 
        ${tables.order_details} AS od ON o.id = od.order_id
      LEFT JOIN 
        ${tables.shift} AS s ON od.shift_id = s.id
      LEFT JOIN 
        ${tables.line} AS l ON o.line_id = l.id
      LEFT JOIN 
        ${tables.product} AS p ON o.product_id = p.id
      LEFT JOIN 
        ${tables.users} AS u1 ON o.created_by = u1.id
      LEFT JOIN 
        ${tables.users} AS u2 ON o.updated_by = u2.id
      Where o.is_deleted = 0
      `;

    const timeCondition = ` (
        (
          STR_TO_DATE(o.expected_start_date, '%Y-%m-%d %H:%i:%s') >= STR_TO_DATE('${start_date}', '%Y-%m-%d %H:%i:%s') 
        ) OR 
        (
          STR_TO_DATE(od.updated_at, '%Y-%m-%d %H:%i:%s') >= STR_TO_DATE('${start_date}', '%Y-%m-%d %H:%i:%s') 
          AND STR_TO_DATE(od.updated_at, '%Y-%m-%d %H:%i:%s') < STR_TO_DATE('${end_date}', '%Y-%m-%d %H:%i:%s') 
        )
      )
    `;   

    // Apply Filters if Any
    if(id){
      selectQuery += ` AND o.id = ${id}`;
    }
    if(is_leds !== undefined){
      selectQuery += ` AND l.is_leds = ${is_leds}`;
    }
    if(start_date && end_date){
      // selectQuery += ` AND (l.is_leds = 1 AND ${timeCondition} OR o.status = 0) `;
      selectQuery += ` AND (
        l.is_leds = 1
        AND (
          o.status IN (0, 1, 3)
          OR (o.status = 2 AND ${timeCondition}) 
        )
      ) `;
    }
    if(status){
      selectQuery += ` OR o.status = ${status}`;
    }

    // order by ID in DESC order
    selectQuery += ` ORDER BY o.id DESC;`;
    
    // Execute Query
    const orders = await performQuery(selectQuery, []);
    const transformedOrders = transformData(orders, line_id);


    // send success response
    return sendResponse(res, transformedOrders, "Orders fetched successfully");

  } catch (error) {
    console.log("Failed to Fetch Orders", error);
    return sendErrorResponse(res, error, "Failed to Fetch Orders", 500);
  }
};

//=====> Update (PUT) API for Order ========
module.exports.updateOrder = async (req, res) => {
    try {
      // Get Input From User
      const {
        id,
        line_id,
        order_no,
        order_description,
        expected_qty,
        qty_unit,
        status,
        product_id,
        expected_start_date,
      } = req.body;

      // Validate required fields
      if (
        !id || !line_id || !product_id ||
        !order_no || !order_description ||
        !expected_qty || !qty_unit
      ) {
        const errorMessage = "All fields are required";
        return sendErrorResponse(res, errorMessage, errorMessage, 400);
      }

      // Get Order Record 
      const existingOrders = await performQuery(
        `SELECT * FROM ${tables.orders} WHERE id = ? AND is_deleted = 0`,
        [id]
      );
      if (existingOrders.length === 0) {
        const errorMessage = "Order not found";
        return sendErrorResponse(res, errorMessage, errorMessage, 404);
      }
      const existingOrder = existingOrders[0];

      // Detect Changes in each fields separately
      const change_order_no = existingOrder.order_no !== order_no;
      const change_line_id = existingOrder.line_id !== line_id;
      const change_product_id = existingOrder.product_id !== product_id;
      const change_order_description = existingOrder.order_description !== order_description;
      const change_expected_qty = existingOrder.expected_qty !== expected_qty;
      const change_qty_unit = existingOrder.qty_unit !== qty_unit;
      const change_status = existingOrder.status !== status;
      const change_expected_start_date = existingOrder.expected_start_date !== expected_start_date;
      const anyChange = change_order_no || change_line_id || change_product_id ||
                        change_order_description || change_expected_qty ||
                        change_qty_unit || change_status || change_expected_start_date;
      if (!anyChange) {
        return sendResponse(res, null, "No changes detected", 200);
      }
      
      // get System Time
      const systemTime = await getSystemTime();
      const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

      if(existingOrder.status == 0){
        // If order is not yet started, allow all fields to be updated
        // check for duplicate order_no
        if (change_order_no) {
          const duplicateOrders = await performQuery(
            `SELECT * FROM ${tables.orders} WHERE order_no = ? AND id <> ? AND is_deleted = 0`,
            [order_no, id]
          );
          if (duplicateOrders.length > 0) {
            const errorMessage = "Another order with this Order No already exists";
            return sendErrorResponse(res, errorMessage, errorMessage, 400);
          }
        }
        if(change_status){
          // Check if there is any other order in progress on the same line
          const inProgressOrders = await performQuery(
            `SELECT * FROM ${tables.orders} WHERE line_id = ? AND status = 1 AND id <> ? AND is_deleted = 0`,
            [line_id, id]
          );
          if (inProgressOrders.length > 0) {
            const errorMessage = "Another order is already in progress on this line";
            return sendErrorResponse(res, errorMessage, errorMessage, 400);
          }
        }

        //===> Update Order Data
        const result = await performQuery(
          `UPDATE ${tables.orders} SET ? WHERE id = ?`,[{
            order_no: order_no || existingOrder.order_no,
            order_description: order_description || existingOrder.order_description,
            line_id: line_id || existingOrder.line_id,
            product_id: product_id || existingOrder.product_id,
            expected_qty: expected_qty || existingOrder.expected_qty,
            qty_unit: qty_unit || existingOrder.qty_unit,
            expected_start_date: expected_start_date || existingOrder.expected_start_date,
            status: status || existingOrder.status,
            updated_by: req.user?.id,
            updated_at: currentTime,
          }, id])

        // Send success response
        return sendResponse(res, result, "Order updated successfully", 200);

      }
      else if(existingOrder.status == 1 || existingOrder.status == 3){
        // If order is in progress, only status can be updated
        const result = await performQuery(
          `UPDATE ${tables.orders} SET status = ?, updated_by = ?, updated_at = ? WHERE id = ?`,
          [status || existingOrder.status, req.user?.id, currentTime, id]
        );

        // Send success response
        return sendResponse(res, result, "Status updated successfully", 200);
      }
      else {
        // If order is completed, no changes allowed
        const errorMessage = "Order is Completed and cannot be changed";
        return sendErrorResponse(res, errorMessage, errorMessage, 400);
      }

    } catch (error) {
      console.error("Failed to Update Orders", error);
      return sendErrorResponse(res, error, "Failed to update order", 500);
    }
  };


function transformData(data, line_id) {
  const map = new Map();
  const lineIds = line_id
    ? (Array.isArray(line_id) ? line_id : [line_id]).map(String)
    : null;

  const filteredData = lineIds
    ? data.filter(row => lineIds.includes(String(row.line_id)))
    : data;

  for (const row of filteredData) {
    // Grouping key: adjust if you want a different grouping logic
    const key = `${row.id}-${row.line_id}-${row.product_code}`;
    if (!map.has(key)) {
      map.set(key, {
        id: row.id,
        line_id: row.line_id,
        line: row.line,
        product_id: row.product_id ?? null, // if exists in your real data
        product_code: row.product_code,
        product_name: row.product_name,
        product_speed: row.product_speed,
        expected_qty: row.expected_qty,
        qty_unit: row.qty_unit,
        order_no: row.order_no,
        order_description: row.order_description,
        status: row.status,
        production: 0,
        scrap: 0,
        shipable_product: 0,
        events: []
      });
    }

    const agg = map.get(key);

    // Sum production & scrap (and shipable_product if you want it aggregated too)
    agg.production += parseFloat(row.production) || 0;
    agg.scrap += parseFloat(row.scrap) || 0;
    agg.shipable_product += (parseFloat(row.production) - parseFloat(row.scrap)) || 0;
    
    if (row.order_detail_id !== null) {
      // Push event-level data
      agg.events.push({
        id: row.order_detail_id,
        order_start: row.order_start,
        order_end: row.order_end,
        shift_id: row.shift_id,
        shift_name: row.shift_name,
        production: row.production,
        scrap: row.scrap,
      });
    }
  }

  return Array.from(map.values());
}
