const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");


module.exports.createProduct = async (req, res) => {
  try {
    // Extract data from the request body
    const { line_id, product_code, product_name, feed_rate, speed} = req.body;

    // validation
    if (!line_id || !product_code || !speed || !product_name || !feed_rate) {
      const errorMessage = "Line Id, Product Name, Speed and Feed Rate are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if product_code already exists
    const productData = await performQuery(
      `SELECT * FROM ${tables.product} WHERE product_code = ? AND line_id = ? AND is_deleted = 0`,
      [product_code, line_id]
    );
    if (productData?.length) {
      return sendErrorResponse(res, "Product is already in use", "Product is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert new product record
    const result = await performQuery(
      `INSERT INTO ${tables.product} SET ?`,
      {
        line_id: Number(line_id),
        product_code,
        product_name,
        feed_rate,
        speed,
        created_at: currentTime,
        created_by: req.user?.id,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }
    );

    // Send the response
    return sendResponse(res, { product_id: result.insertId }, "Product created successfully", 201);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while creating Product");
  }
};

module.exports.updateProduct = async (req, res) => {
  try {
    // Extract data from the request body
    const { id, line_id, product_code, product_name, feed_rate, speed, is_active} = req.body;

    // validation
    if (!id || !line_id || !product_code || !speed || !product_name || !feed_rate) {
      const errorMessage = "Id, Line Id, Product Name, Speed and Feed Rate are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if product_code already exists
    const productData = await performQuery(
      `SELECT * FROM ${tables.product} WHERE product_code = ? AND line_id = ? AND id <> ? AND is_deleted = 0`,
      [product_code, line_id, id]
    );
    if (productData?.length) {
      return sendErrorResponse(res, "Product is already in use", "Product is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update product record
    const result = await performQuery(
      `UPDATE ${tables.product} SET ? WHERE id = ?`,[{
        line_id: Number(line_id),
        product_code,
        product_name,
        feed_rate,
        speed,
        is_active: is_active !== undefined ? is_active : 1,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }, id]);

    // Send the response
    return sendResponse(res, result, "Product updated successfully", 201);

  } catch (error) {
    console.log("error > ", error);
    return sendErrorResponse(res, error, "Error while Updating Product");
  }
};

module.exports.getProducts = async (req, res) => {
  try {
    // Get Filter Options From Query Params
    const { id, line_id, product_code } = req.query;

    // Build Query
    let query = `
      SELECT 
        product.id,  
        product.line_id,
        line.line,
        product.product_code,
        product.product_name,
        product.feed_rate,
        product.speed
      FROM ${tables.product}
      LEFT JOIN ${tables.line} ON product.line_id = line.id
      WHERE product.is_deleted = 0`;

    // Add Filters
    if (id) {
      query += ` AND product.id = ${id}`;
    }
    if (line_id) {
      query += ` AND product.line_id = ${line_id}`;
    }
    if (product_code) {
      query += ` AND product.product_code = '${product_code}'`;
    }

    // Final Query Ordering
    query += ` ORDER BY product.id DESC`;

    // Execute Query
    const products = await performQuery(query);

    // Send Response
    return sendResponse(res, products, "Products fetched successfully", 200);
  
  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching Product");
  }
};

module.exports.deleteProduct = async (req, res) => {
  try {
    const { id } = req.query;

    if (!id) {
      const errorMessage = "Id is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft Delete product record
    const result = await performQuery(
      `UPDATE ${tables.product} SET ? WHERE id = ?`,
      [{
        is_deleted: 1,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }, id]
    );

    // Send the response
    return sendResponse(res, result, "Product deleted successfully", 200);
    
  } catch (error) {
    return sendErrorResponse(res, error, "Error while deleting Product");
  }
};
