const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const { getCurrentShift } = require("../helper/shift.helper.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");


module.exports.createShift = async (req, res) => {
  try {
    // Extract data from request body
    const { shift_name, shift_start, shift_end, is_active} = req.body;

    // Validate required fields
    if (!shift_name || !shift_start || !shift_end) {
      const error = new Error("shift_name and Start/ End Time are required");
      return sendErrorResponse(res, error, "shift_name and Start/ End Time are required");
    }

    // Check for existing shift
    const existingShift = await performQuery(
      `SELECT * FROM ${tables.shift} WHERE shift_name = ? AND shift_start = ? AND shift_end = ? AND is_deleted = 0`,
      [shift_name, shift_start, shift_end]
    );
    if (existingShift?.length) {
      const errorMessage = "Shift is already in use";
      return sendErrorResponse(res, errorMessage, errorMessage, 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert new shift into the database
    const result = await performQuery(
      `INSERT INTO ${tables.shift} SET ? `,
      {
        shift_name,
        shift_start,
        shift_end,
        created_by: req.user?.id,
        created_at: currentTime,
        updated_by: req.user?.id,
        updated_at: currentTime,
        is_active: is_active !==undefined? is_active : 1,
      }
    );

    // Send the response
    return sendResponse(res, { shift_id: result.insertId }, "Shift created successfully", 201);

  } catch (error) {
    console.log("Error While Creating Shift: ", error);
    return sendErrorResponse(res, error, "Error while creating Shift", 409);
  }
};

module.exports.updateShift = async (req, res) => {
  try {
    // Extract data from request body
    const { shift_name, shift_start, shift_end, id, is_active } = req.body;

    // Validate required fields
    if (!shift_name || !shift_start || !shift_end || !id ) {
      const error = new Error("Shift and id is required");
      return sendErrorResponse(res, error, "Shift and id is required");
    }

    // Check for existing shift
    const existingShift = await performQuery(
      `SELECT * FROM ${tables.shift} WHERE shift_name = ? AND shift_start = ? AND shift_end = ? AND id <> ? AND is_deleted = 0`,
      [shift_name, shift_start, shift_end, id]
    );
    if (existingShift?.length) {
      const errorMessage = "Shift is already in use";
      return sendErrorResponse(res, errorMessage, errorMessage, 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");
    
    // Update shift in the database
    const updateShift = await performQuery(
      `UPDATE ${tables.shift} SET ? WHERE id = ?`,[{
        shift_name,
        shift_start: moment(shift_start).format("HH:mm"),
        shift_end: moment(shift_end).format("HH:mm"),
        is_active: is_active || 1,
        updated_by: req.user?.id,
        updated_at: currentTime,
      }, id]
    )

    // Send the response
    return sendResponse(res, { shift_id: id }, "Shift updated successfully", 201);

  } catch (error) {
    console.log("Error While Updating Shift: ", error);
    return sendErrorResponse(res, error, "Error while updating Shift");
  }
};

module.exports.getShifts = async (req, res) => {
  try {
    // Get Filter Parameters from Query 
    const { id, is_active } = req.query;

    // Build the query
    let query = `
      SELECT 
        id, shift_name, shift_start, shift_end, is_active 
      FROM ${tables.shift} 
      WHERE is_deleted = 0`;

    // Apply Filters
    if (id) {
      query += ` AND id = ${id}`;
    }
    if (is_active) {
      query += ` AND is_active = ${is_active}`;
    }

    // Sorting 
    query += ` order by shift_start`;

    // Execute the query
    const shiftRecords = await performQuery(query);

    const result = await getCurrentShift(shiftRecords);
    
    // Send the response
    return sendResponse(res, result, "Fetch Shift data successfully", 200);

  } catch (error) {
    console.log("Error while fetching Shift: ", error);
    return sendErrorResponse(res, error, "Error while fetching Shift");
  }
};

module.exports.deleteShift = async (req, res) => {
  try {
    // Get id from query parameters
    const { id } = req.query;

    // Validate id
    if (!id) {
      const error = new Error("Id is required");
      return sendErrorResponse(res, error, "Id is required");
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft delete the shift
    const result = await performQuery(
      `UPDATE ${tables.shift} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_by: req.user?.id,
          updated_at: currentTime,
        },
        id,
      ]
    );

    // Send the response
    return sendResponse(res, { shift_id: id }, "Shift deleted successfully", 200);
    
  } catch (error) {
    console.log("Error while deleting Shift: ", error);
    return sendErrorResponse(res, error, "Error while deleting Shift");
  }
};
