const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");

module.exports.createSubEquipment = async (req, res) => {
  try {
    // Extract data from the request body
    const { equipment_id, sub_equipment } = req.body;

    // validation
    if (!equipment_id || !sub_equipment) {
      const errorMessage = "Equipment ID and sub_equipment are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if sub_equipment already exists
    const subEquipmentData = await performQuery(
      `SELECT * FROM ${tables.sub_equipment} WHERE sub_equipment = ? and equipment_id = ? and is_deleted = 0`,
      [sub_equipment, equipment_id]
    );
    if (subEquipmentData?.length) {
      return sendErrorResponse(res, "sub_equipment is already in use", "sub_equipment is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert new sub_equipment record
    const result = await performQuery(
      `INSERT INTO ${tables.sub_equipment} SET ? `,
      {
        equipment_id: Number(equipment_id),
        sub_equipment,
        created_at: currentTime,
        created_by: req.user?.id,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }
    );

    // Send the response
    return sendResponse(res, { sub_equipment_id: result.insertId }, "sub_equipment created successfully", 201);

  } catch (error) {
    console.log("Error While Creating sub_equipment > ", error);
    return sendErrorResponse(res, error, "Error while creating sub_equipment");
  }
};

module.exports.getSubEquipments = async (req, res) => {
  try {
    // Get Filter parameters from the request query
    const { id, department_id, line_id, equipment_id } = req.query;

    // Build the base select query
    let selectSubEquipmentQuery = `
      SELECT 
        sub_equip.id, sub_equip.sub_equipment, sub_equip.equipment_id, equip.equipment, ln.id AS line_id, ln.line
      FROM ${tables.sub_equipment} sub_equip
      LEFT JOIN ${tables.equipment} equip ON sub_equip.equipment_id = equip.id
      LEFT JOIN ${tables.line} ln ON equip.line_id = ln.id
      LEFT JOIN ${tables.department} dept ON ln.department_id = dept.id
      LEFT JOIN ${tables.users} usr ON equip.owner_id = usr.id
      WHERE sub_equip.is_deleted = 0
    `;
    
    // Apply Filters if provided
    if(id){
      selectSubEquipmentQuery += ` AND sub_equip.id=${id}`;
    }
    if(department_id){
      selectSubEquipmentQuery += ` AND dept.id=${department_id}`;
    }
    if(line_id){
      selectSubEquipmentQuery += ` AND ln.id=${line_id}`;
    }
    if(equipment_id){
      selectSubEquipmentQuery += ` AND equip.id=${equipment_id}`;
    }

    // Sort Data by ID in Descending Order
    selectSubEquipmentQuery += ` ORDER BY sub_equip.id DESC`;
    
    // Perform the database query
    const sub_equipments = await performQuery(selectSubEquipmentQuery);

    // Send the response
    return sendResponse(res, sub_equipments, "sub_equipment fetched successfully");

  } catch (error) {
    console.log("Error While Fetching sub_equipment > ", error);
    return sendErrorResponse(res, error, "Error while fetching sub_equipment");
  }
};

module.exports.updateSubEquipment = async (req, res) => {
  try {
    // Extract data from the request body
    const { id, equipment_id, sub_equipment } = req.body;

    // validation
    if (!equipment_id || !sub_equipment || !id) {
      const errorMessage = "Department and id is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if sub_equipment already exists
    const subEquipmentData = await performQuery(
      `SELECT * FROM ${tables.sub_equipment} WHERE sub_equipment = ? and equipment_id = ? and id <> ? and is_deleted = 0`,
      [sub_equipment, equipment_id, id]
    );
    if (subEquipmentData?.length) {
      return sendErrorResponse(res, "sub_equipment is already in use", "sub_equipment is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update sub_equipment record
    const result = await performQuery(
      `UPDATE ${tables.sub_equipment} SET ? WHERE id = ? `, [{
        equipment_id: Number(equipment_id),
        sub_equipment,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }, id]
    );

    // Send the response
    return sendResponse(res, result, "sub_equipment updated successfully", 201);

  } catch (error) {
    console.log("Error While Updating sub_equipment > ", error);
    return sendErrorResponse(res, error, "Error while Updating sub_equipment");
  }
};

module.exports.deleteSubEquipment = async (req, res) => {
  try {
    const { id } = req.query;

    // validation
    if (!id) {
      const errorMessage = "Id is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft Delete sub_equipment record
    const result = await performQuery(
      `UPDATE ${tables.sub_equipment} SET ? WHERE id = ? `, [{
        is_deleted: 1,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }, id]
    );

    // Send the response
    return sendResponse(res, result, "sub_equipment deleted successfully", 200);

  } catch (error) {
    console.log("Error While Deleting sub_equipment > ", error);
    return sendErrorResponse(res, error, "Error while deleting sub_equipment");
  }
};
