const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");


module.exports.createTeam = async (req, res) => {
  try {
    // Extract data from the request body
    const { team} = req.body;

    // validation
    if (!team) {
      const errorMessage = "Team Name is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if team already exists
    const teamData = await performQuery(
      `SELECT * FROM ${tables.teams} WHERE team = ? AND is_deleted = 0`,
      [team]
    );
    if (teamData?.length) {
      return sendErrorResponse(res, "Team is already in use", "Team is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert new team record
    const result = await performQuery(
      `INSERT INTO ${tables.teams} SET ?`,
      {
        team,
        created_at: currentTime,
        created_by: req.user?.id,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }
    );

    // Send the response
    return sendResponse(res, { team_id: result.insertId }, "Team created successfully", 201);

  } catch (error) {
    console.log("Error While Creating Team: ", error);
    return sendErrorResponse(res, error, "Error while creating team", 409);
  }
};

module.exports.updateTeam = async (req, res) => {
  try {
    // Extract data from the request body
    const { id, team, is_active} = req.body;

    // validation
    if (!id || !team) {
      const errorMessage = "Team ID and Team Name are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check if team already exists
    const teamData = await performQuery(
      `SELECT * FROM ${tables.teams} WHERE team = ? AND id != ? AND is_deleted = 0`,
      [team, id]
    );
    if (teamData?.length) {
      return sendErrorResponse(res, "Team is already in use", "Team is already in use", 409);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update team record
    await performQuery(
      `UPDATE ${tables.teams} SET ? WHERE id = ?`,
      [{
        team,
        is_active,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }],
      id
    );

    // Send the response
    return sendResponse(res, {}, "Team updated successfully", 200);

  } catch (error) {
    console.log("Error While Updating Team: ", error);
    return sendErrorResponse(res, error, "Error while updating team");
  }
};

module.exports.getTeams = async (req, res) => {
  try {
    // Get Filter params from request query
    const { id, is_active } = req.query;

    // Build the query dynamically based on filters
    let query = `SELECT * FROM ${tables.teams} WHERE is_deleted = 0`;
    
    // Add filters if provided
    if (id) {
      query += ` AND id = ${Number(id)}`;
    }
    if (is_active !== undefined) {
      query += ` AND is_active = ${Number(is_active)}`;
    }

    // Sort by id descending
    query += ` ORDER BY id DESC`;

    // Execute the query
    const teams = await performQuery(query);

    // Send the response
    return sendResponse(res, teams, "Teams fetched successfully");

  } catch (error) {
    console.log("Error While Fetching Teams: ", error);
    return sendErrorResponse(res, error, "Error while fetching Teams");
  }
};

module.exports.deleteTeam = async (req, res) => {
  try {
    // Extract team ID from request Query
    const { id } = req.query;

    // Validation
    if (!id) {
      const errorMessage = "Team ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft delete the team in the database
    await performQuery(
      `UPDATE ${tables.teams} SET ? WHERE id = ?`,
      [{
        is_deleted: 1,
        updated_at: currentTime,
        updated_by: req.user?.id,
      }],
      id
    );

    // Send the response
    return sendResponse(res, {}, "Team deleted successfully");

  } catch (error) {
    console.log("Error While Deleting Team: ", error);
    return sendErrorResponse(res, error, "Error while deleting team");
  }
};
