const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");


module.exports.createUnit = async (req, res) => {
  try {
    // Extract data from request body
    const { unit } = req.body;

    // Validate required fields
    if (!unit) {
      const errorMessage = "Unit Name is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for existing unit
    const existingUnit = await performQuery(
      `SELECT * FROM ${tables.unit} WHERE unit = ? AND is_deleted = 0`,
      [unit]
    );
    if (existingUnit && existingUnit.length > 0) {
      const errorMessage = "Unit already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Insert new unit into the database
    const result = await performQuery(
      `INSERT INTO ${tables.unit} SET ? `,
      {
        unit,
        created_by: req.user?.id,
        created_at: currentTime,
        updated_by: req.user?.id,
        updated_at: currentTime,
      }
    );

    // Send the response
    return sendResponse(res, { unit_id: result.insertId }, "Unit created successfully", 201);
    
  } catch (error) {
    console.log("Error While Creating Unit: ", error);
    return sendErrorResponse(res, error, "Error while creating Unit", 409);
  }
};

module.exports.updateUnit = async (req, res) => {
  try {
    // Extract data from request body
    const { id, unit, is_active } = req.body;

    // Validate required fields
    if (!id || !unit) {
      const errorMessage = "Unit ID and Unit Name are required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // Check for existing unit
    const existingUnit = await performQuery(
      `SELECT * FROM ${tables.unit} WHERE unit = ? AND id <> ? AND is_deleted = 0`,
      [unit, id]
    );
    if (existingUnit && existingUnit.length > 0) {
      const errorMessage = "Unit already exists";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Update unit in the database
    await performQuery(
      `UPDATE ${tables.unit} SET ? WHERE id = ?`,
      [
        {
          unit,
          is_active: is_active !== undefined ? is_active : 1,
          updated_by: req.user?.id,
          updated_at: currentTime,
        },
        id,
      ]
    );

    // Send the response
    return sendResponse(res, null, "Unit updated successfully");

  } catch (error) {
    console.log("Error While Updating Unit: ", error);
    return sendErrorResponse(res, error, "Error while updating Unit");
  }
};

module.exports.getUnits = async (req, res) => {
  try {
    // Get Filter Parameters from Query
    const { id, is_active } = req.query;

    // Build the query
    let query = `SELECT id, unit, is_active FROM ${tables.unit} WHERE is_deleted = 0`;

    // Apply Filters
    if (id) {
      query += ` AND id = ${parseInt(id, 10)}`;
    }
    if (is_active !== undefined) {
      query += ` AND is_active = ${parseInt(is_active, 10)}`;
    }

    // Sort the results
    query += ` ORDER BY id DESC`;

    // Execute the query
    const units = await performQuery(query);

    // Send the response
    return sendResponse(res, { units }, "Units fetched successfully");

  } catch (error) {
    console.log("Error While Fetching Units: ", error);
    return sendErrorResponse(res, error, "Error while fetching Units");
  }
};


module.exports.deleteUnit = async (req, res) => {
  try {
    // Extract unit ID from request Query
    const { id } = req.query;

    // Validate required fields
    if (!id) {
      const errorMessage = "Unit ID is required";
      return sendErrorResponse(res, errorMessage, errorMessage);
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Soft delete the unit in the database
    await performQuery(
      `UPDATE ${tables.unit} SET ? WHERE id = ?`,
      [
        {
          is_deleted: 1,
          updated_by: req.user?.id,
          updated_at: currentTime,
        },
        id,
      ]
    );

    // Send the response
    return sendResponse(res, null, "Unit deleted successfully");

  } catch (error) {
    console.log("Error While Deleting Unit: ", error);
    return sendErrorResponse(res, error, "Error while deleting Unit");
  }
};
