const { sendErrorResponse, sendResponse } = require("../utils/index.js");
const tables = require("../utils/tables.js");
const { performQuery } = require("../utils/db.js");
const bcrypt = require("bcryptjs");
const { sendMail } = require("../services/MAIL.js");
const { getSystemTime } = require("../functions/getTimezone.js");
const moment = require("moment-timezone");


module.exports.createUser = async (req, res) => {
  try {
    const { 
      name, 
      email,
      is_admin,
      role, 
      department_id, 
      line_id,
      team_id, 
      designation_id, 
    } = req.body;

    if (!name || !email) {
      const error = new Error("Name and Email are required");
      return sendErrorResponse(
        res,
        error,
        "Name and Email are required"
      );
    }

    const selectUserQuery = `SELECT email FROM ${tables.users} WHERE email = ? AND is_deleted=0`;
    const selectParams = [email];

    // Perform the database query
    const user = await performQuery(selectUserQuery, selectParams);
    if (user?.length) {
      const error = new Error("Email address is already in use");
      return sendErrorResponse(res, error, "Email address is already in use");
    } 
    
    // Generate Random Password and Hash it
    const password = Math.random().toString(36).slice(-8);
    const hashedPassword = await bcrypt.hash(password, 10);

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    // Create User
    const insertQuery = `INSERT INTO ${tables.users} SET ?`;
    const insertParams = {
      name: name,
      email: email,
      password: hashedPassword,
      is_admin: is_admin || 0,
      role: role || null,
      department_id: department_id || null,
      line_id: line_id || null,
      team_id: team_id || null,
      designation_id: designation_id || null,
      is_new: 1,
      created_by: req.user?.id || null,
      created_at: currentTime,
      updated_by: req.user?.id || null,
      updated_at: currentTime,
    };
    const result = await performQuery(insertQuery, insertParams);
    
    // Send Email to User with Password - TODO
    const body = buildWelcomeEmail({ name, email, tempPassword: password });
    await sendMail(email, "Welcome to Our Service", body);

    // Log the user creation event
    return sendResponse(res, {id: result.insertId}, "User created successfully", 201);

  } catch (error) {
    console.log("Error While Creating User > ", error);
    return sendErrorResponse(res, error, "Error while creating User");
  }
};

module.exports.updateUser = async (req, res) => {
  try {
    const {
      id, 
      name, 
      is_admin,
      role, 
      department_id, 
      line_id,
      team_id, 
      designation_id, 
      is_active,
    } = req.body;

    
    if (!id) {
      const error = new Error("Id is required");
      return sendErrorResponse(
        res,
        error,
        "Id is required"
      );
    }

    // Perform the database query
    const user = await performQuery(`SELECT * FROM ${tables.users} WHERE id = ? AND is_deleted=0`, [id]);
    // console.log("user > ", user);
    if (!user?.length) {
      return sendErrorResponse(res, "User not found", "User not found");
    }
    const currentUser = user[0];

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    const result = await performQuery(
      `UPDATE ${tables.users} SET ? WHERE id = ?`, [{
      name: name || currentUser.name,
      is_admin: is_admin !== undefined ? is_admin : currentUser.is_admin,
      role: role || currentUser.role,
      department_id: department_id || currentUser.department_id,
      line_id: line_id || currentUser.line_id,
      team_id: team_id || currentUser.team_id,
      designation_id: designation_id || currentUser.designation_id,
      is_active: is_active !== undefined ? is_active : currentUser.is_active,
      updated_by: req.user?.id || null,
      updated_at: currentTime,
    }, id]);

    // Send Response
    return sendResponse(res, { affectedRows: result.affectedRows }, "User updated successfully", 200);

  } catch (error) {
    console.log("Error While Updating User > ", error);
    return sendErrorResponse(res, error, "Error while updating User");
  }
};

module.exports.getUsers = async (req, res) => {
  try {
    // gET Filter Parameters
    const { id, department_id, line_id, team_id, designation_id } = req.query;

    let selectUserQuery = `
      SELECT 
        users.id, users.name, users.email, users.is_admin, 
        users.profile_url, users.role, users.is_active, 
        users.department_id, users.line_id, users.team_id, users.designation_id, 
        department.department, line.line, teams.teams, designations.designation
      FROM 
        ${tables.users} 
      LEFT JOIN 
        ${tables.department} ON department.id = users.department_id
      LEFT JOIN 
        ${tables.line} ON line.id = users.line_id
      LEFT JOIN 
        ${tables.teams} ON teams.id = users.team_id 
      LEFT JOIN 
        ${tables.designations} ON designations.id = users.designation_id 
      WHERE
        users.is_deleted = 0
    `;

    // Append filters to the query
    if (id) {
      selectUserQuery += ` AND users.id = ${id}`;
    }
    if (department_id) {
      selectUserQuery += ` AND users.department_id = ${department_id}`;
    }
    if (line_id) {
      selectUserQuery += ` AND users.line_id = ${line_id}`;
    }
    if (team_id) {
      selectUserQuery += ` AND users.team_id = ${team_id}`;
    }
    if (designation_id) {
      selectUserQuery += ` AND users.designation_id = ${designation_id}`;
    }

    // Order By id DESC
    selectUserQuery += ` ORDER BY users.id DESC`;

    // Perform the database query
    const user = await performQuery(selectUserQuery);
    
    // Send Response
    return sendResponse(res, user, "Fetch User data successfully", 200);

  } catch (error) {
    return sendErrorResponse(res, error, "Error while fetching User");
  }
};

module.exports.deleteUser = async (req, res) => {
  try {
    const { id } = req.query;

    // validation
    if (!id) {
      return sendErrorResponse(
        res,
        "Id is required",
        "Id is required"
      );
    }

    // get System Time
    const systemTime = await getSystemTime();
    const currentTime = moment(systemTime).format("YYYY-MM-DD HH:mm:ss");

    const result = await performQuery(
      `UPDATE ${tables.users} SET ? WHERE id = ?`, [{
      is_deleted: 1,
      updated_by: req.user?.id || null,
      updated_at: currentTime,
    }, id]);

    // Send Response
    return sendResponse(res, { affectedRows: result.affectedRows }, "User deleted successfully", 200);

  } catch (error) {
    console.log("Error While Deleting User > ", error);
    return sendErrorResponse(res, error, "Error while deleting User");
  }
};



// emailTemplate.js
function buildWelcomeEmail({ name, email, tempPassword }) {
  return `
  <!DOCTYPE html>
  <html lang="en" style="font-family: system-ui, -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;">
    <head>
      <meta charset="UTF-8" />
      <title>Welcome to Shan Foods LEDS System</title>
    </head>
    <body style="background-color:#f5f7fa; padding:24px; margin:0;">
      <table width="100%" border="0" cellspacing="0" cellpadding="0" style="max-width:480px; margin:0 auto; background:#ffffff; border-radius:12px; box-shadow:0 8px 24px rgba(0,0,0,0.07); overflow:hidden; border:1px solid #e5e7eb;">
        <tr>
          <td style="background:linear-gradient(90deg,#4f46e5,#6366f1); padding:20px 24px; color:#fff; text-align:left;">
            <div style="font-size:14px; font-weight:500; letter-spacing:-0.03em; opacity:.9;">Shan Foods LEDS System</div>
            <div style="font-size:18px; font-weight:600; margin-top:4px;">Your account is ready 🎉</div>
          </td>
        </tr>

        <tr>
          <td style="padding:24px 24px 8px 24px; color:#111827; font-size:16px; line-height:1.5;">
            Hi <strong>${name}</strong>,
            <br/><br/>
            Your Shan Foods LEDS System account has been created. Here are your temporary login details:
          </td>
        </tr>

        <tr>
          <td style="padding:0 24px 16px 24px;">
            <table width="100%" cellpadding="0" cellspacing="0" style="background:#f9fafb; border:1px solid #e5e7eb; border-radius:8px; font-size:15px; line-height:1.5; color:#111827;">
              <tr>
                <td style="padding:12px 16px; width:30%; font-weight:600; white-space:nowrap; border-bottom:1px solid #e5e7eb;">Email</td>
                <td style="padding:12px 16px; border-bottom:1px solid #e5e7eb;">${email}</td>
              </tr>
              <tr>
                <td style="padding:12px 16px; width:30%; font-weight:600; white-space:nowrap;">Temp Password</td>
                <td style="padding:12px 16px; font-family:ui-monospace, SFMono-Regular, Menlo, Consolas, monospace; background:#fffbea; border:1px solid #facc15; border-radius:6px; display:inline-block;">
                  ${tempPassword}
                </td>
              </tr>
            </table>
          </td>
        </tr>

        <tr>
          <td style="padding:0 24px 24px 24px; font-size:14px; line-height:1.5; color:#6b7280;">
            For security, please sign in and change this password immediately.
          </td>
        </tr>

        <tr>
          <td style="padding:0 24px 32px 24px; text-align:center;">
            <a href="https://opex3-shanfoods.oee-tracker.com/login"
               style="background:#4f46e5; color:#fff; text-decoration:none; font-size:15px; font-weight:600; padding:12px 20px; border-radius:8px; display:inline-block;">
              Login →
            </a>
          </td>
        </tr>

        <tr>
          <td style="padding:20px 24px; background:#f9fafb; font-size:12px; line-height:1.5; color:#9ca3af; text-align:center; border-top:1px solid #e5e7eb;">
            If you didn’t request this account, you can ignore this email.<br/>
            © 2025 Pakistan Automation Inc.
          </td>
        </tr>
      </table>
    </body>
  </html>
  `;
}
