const jwt = require("jsonwebtoken");
const tables = require("../utils/tables");
const { performQuery } = require("../utils/db");
const { JWT_SECRET } = require("../config");
const { sendErrorResponse } = require("../utils");
const { logger } = require("../helper/logger.helper");

// Define the middleware
module.exports.authMiddleware = async (req, res, next) => {
  try {
    console.log("------------Request Query--------", req.query);
    console.log("------------Request Body--------", req.body);
    console.log("------------Request Path--------", req.path);
    console.log("------------Request method--------", req.method);
    const { method, url, headers, body, query, path } = req; // Extract relevant request data
    const { statusCode, statusMessage } = res; // Extract relevant response data
    const { token, authorization, ...rest } = headers;
    const authExceptionConditions = 
      req.path == '/auth/login' || req.path == '/auth/signup' || 
      req.path == '/auth/forgot-password' || req.path == '/auth/reset-password' || req.path == '/auth/set-password';
            
    let getToken = token?.split(" ")?.[1] || authorization?.split(" ")?.[1] || '';
    if (getToken.startsWith('"') && getToken.endsWith('"')) {
      getToken = getToken.slice(1, -1);
    }
    let userData = {};
    let error = {}
    if (getToken && !authExceptionConditions) {
      jwt.verify(getToken, JWT_SECRET, async (err, user) => {
        if (err) error = err;
        userData = user;
      });
    }
    else if (!authExceptionConditions) {
      const err = new Error("Not Authenticated");
      err.statusCode = 401;
      err.message = "User Not Authenticated";
      return sendErrorResponse(res, err.message, err.message, err.statusCode);
    }

    const issueDate = userData?.iat * 1000;
    const nowDate = new Date().getTime();
    const diff = nowDate - issueDate;;
    
    if (!authExceptionConditions) {
      if (error.name === 'TokenExpiredError') {
        return sendErrorResponse(res, error, "Token has expired", 401);
      }

      // Check if user is valid
      const userRecord = await performQuery(
        `SELECT * FROM ${tables.users} WHERE id = ? AND is_active = 1 AND is_deleted = 0`, 
        [userData.id]
      );
      
      // If the user does not exist, send an error response
      if (!userRecord) {  
        return sendErrorResponse(res, {}, "Invalid token", 401);
      }
      userData = userRecord?.[0];
    }
      
    req.user = {
      id: userData.id,
      name: userData.name,
      email: userData.email,
      role: userData.role,
      is_admin: userData.is_admin,
    };
      
    logger.info(`Request: ${method} ${url}`, {
      headers: rest,
      path,
      body,
      query,
      statusCode,
      statusMessage,
      user: {
        id: userData?.id,
        name: userData?.name,
        is_admin: userData?.is_admin,
      },
    });

    
    
    if (diff > 3600000) {
      return sendErrorResponse(res, {}, "Token expired, You have been logged out", 401);
    }
    // Generate a JWT token for the user
    const payload = {
      id: userData.id,
      name: userData.name,
      email: userData.email,
      role: userData.role,
      is_admin: userData.is_admin,
      iat: Math.floor(Date.now() / 1000),
    };

    // Set new token in response header
    const newToken = await jwt.sign(payload, JWT_SECRET, {
      expiresIn: "6h",
    });
    res.set('new-token', newToken);
  
    next();
  } catch (error) {
    console.log("Error: ", error.message);
    return sendErrorResponse(res, error, "Invalid token", 401);
  }
};

module.exports.adminAuth = async (req, res, next) => {
  if(req.user && req.user.is_admin){
    next();
  } else {
    return sendErrorResponse(res, {}, "Admin Privileges Required", 403);
  }
};

